%{
This m-file belongs to Chapter 5 of the dissertation 'Electrochemical CO2
conversion with a flow-through copper hollow fibre'
Anne Sustronk (1,2)
(1) Films in Fluids, Department of Science and Technology, University of 
Twente, Enschede, The Netherlands
(2) Photocatalytic Synthesis Group, Department of Science and Technology, 
University of Twente, Enschede, The Netherlands

The file is used to prepare Table 5.1 in the chapter.

This file is called by PlotWetDry.m and requires CalcErr_WetDry.m for
execution.

Brief overview of experimental conditions: 10 ml/min CO2 through fibre and 
20 ml/min CO2 through head space, 0.5M KHCO3, other conditions same as 
in Chapter 4.

Version date m-file: March 22, 2023
%}

function [Data] = WetDry(Data)
warning('off','MATLAB:table:ModifiedAndSavedVarnames')

% Rename input parameters
Location = Data.Location;
Folder1 = Data.Folder1;
FileNameEC = Data.FileNameEC;
Area = Data.Area;
AreaErr = Data.AreaErr;
ESet = Data.ESet;
VarName = Data.VarName;

% Some constants
Vm = 22.4;              % Molar volume (mL/mol)
FlowRate = 30/60;       % Flow rate of gas to gas chromatograph (ml/s)
F = 96485;              % Faraday constant (C/mol)


%% Calibration input

% Hydrogen
XCalH2 = Data.XCalH2;
YCalH2 = Data.YCalH2;

% Carbon monoxide
XCalCO = Data.XCalCO;
YCalCO = Data.YCalCO;

% Standard deviation calibration
H2ppmErr = Data.H2ppmErr;
COppmErr = Data.COppmErr;

%% File path resistance values
FilePathR =  append(Location, '\resistances.txt');
FilePathR = cell2mat(FilePathR);  % Convert cell array with file paths to matrix
LoadDataResistance = importdata(FilePathR);

%% Calculations
for j = 1:length(ESet)
    
    % Retrieve file path potentiostat data
    FilePathEC = append(Location, Folder1(j,:), FileNameEC(j,:));
    FilePathEC = cell2mat(FilePathEC);  % Convert cell array with file paths to matrix

    %% Load data from potentiostat
    LoadDataEC = importdata(FilePathEC);

    % LoadDataEC is a struct containing fields for every investigated potential
    % The columns refer to: time/s, <I>/mA, Ewe/V, <Ece>/V, dQ/C, Rcmp/Ohm,
    % P/W, Ewe-Ece/V
    Data.(VarName(j,:)).EC = LoadDataEC.data;

    % Recalculate time (time does not start at t=0)
    tElapsed = Data.(VarName(j,:)).EC(:,1);
    tStart = Data.(VarName(j,:)).EC(1,1);
    Data.(VarName(j,:)).t = tElapsed - tStart;

    % Total current
    Data.(VarName(j,:)).I = Data.(VarName(j,:)).EC(:,2);

    % Calculate current density
    Data.(VarName(j,:)).CD = Data.(VarName(j,:)).EC(:,2)/Area(j);

    % Calculate average current (density) over the last 30 minutes of the
    % experiment
    findTime = find(Data.(VarName(j,:)).t >= 30*60);

    q1 = size(findTime);

    % Calculate average current and estimate corresponding error
    Data.Iavg30mins(j,1) = mean(Data.(VarName(j,:)).I(findTime(:,1),1));
    Data.Iavg30mins(j,2) = sqrt(sum((Data.(VarName(j,:)).I(findTime(:,1),1)-Data.Iavg30mins(j,1)).^2)/(q1(1,1)-1));

    % Calculate average current density and estimate corresponding error
    Data.CDavg30mins(j,1) = mean(Data.(VarName(j,:)).CD(findTime(:,1),1));
    Data.CDavg30mins(j,2) = sqrt((Data.Iavg30mins(j,2)/Data.Iavg30mins(j,1))^2 + ((AreaErr(j)/Area(j))^2))*Data.CDavg30mins(j,1);


    %% TCD file paths
    FileNameTCD = Data.FileNameTCD;
    FilePathTCD = append(Location, Folder1, FileNameTCD(j,:));
    FilePathTCD = cell2mat(FilePathTCD);  % Convert cell array with file paths to matrix

    %% Obtain H2 current and H2 partial current density
    LoadDataTCD = readtable(FilePathTCD(j,:));
    LoadDataTCD = LoadDataTCD(:,7:end);

    Data.(VarName(j,:)).TCD = LoadDataTCD;

    TCDArea = table2array(Data.(VarName(j,:)).TCD);

    % Calculate average area of injections 5, 6 and 7
    Data.TCD_avg(j,:) = mean(TCDArea(5:end,:));
    
    % Calculate H2 concentration and corresponding error
    Data.ConcAndErr(j,:) = CalcErr_WetDry(XCalH2, YCalH2, H2ppmErr,Data.TCD_avg(j,1));

    % Calculate corresponding current
    Data.IH2(j,:) = ((((Data.ConcAndErr(j,:))/10000/100)*(FlowRate/Vm))*-2*F);

    % Calculate corresponding current density
    Data.CDH2(j,1) = Data.IH2(j,1)/Area(j);

    % Calculate error in current density
    Data.CDH2(j,2) = sqrt(((Data.IH2(j,2))/Data.IH2(j,1))^2+(AreaErr(j)/Area(j))^2)*(Data.CDH2(j,1));

    %% Obtain CO current and CO partial current density

    % Calculate CO concentration and corresponding error
    Data.ConcAndErrCO(j,:) = CalcErr_WetDry(XCalCO, YCalCO, COppmErr,Data.TCD_avg(j,4));

    % Calculate corresponding current
    Data.ICO(j,:) = ((((Data.ConcAndErrCO(j,:))/10000/100)*(FlowRate/Vm))*-2*F);

    % Calculate corresponding current density
    Data.CDCO(j,1) = Data.ICO(j,1)/Area(j);

    % Calculate error in current density
    Data.CDCO(j,2) = sqrt(((Data.ICO(j,2))/Data.ICO(j,1))^2+(AreaErr(j)/Area(j))^2)*(Data.ICO(j,1));

    %% Calculate actual potential at working electrode and error in the potential

    Data.R(j,:) = LoadDataResistance(j,:);

    RAvg = mean(Data.R(j,:));

    % Actual potentail
    Data.Ereal(j,1) = ESet(j)- mean((Data.(VarName(j,:)).EC(findTime(:,1),2))/1000)*RAvg;

    % Error calculations
    RErr = Data.R(j,1) - RAvg;

    %Error in potential
    Data.Ereal(j,2) =  mean((Data.(VarName(j,:)).EC(findTime(:,1),2))/1000)*RErr;

    %% Calculate current that is not accounted for 

    Data.Unaccounted(j,1) = Data.CDavg30mins(j,1) - Data.CDH2(j,1) - Data.CDCO(j,1);
end
end